/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.pagememory.persistence.store;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.ignite.internal.failure.FailureContext;
import org.apache.ignite.internal.failure.FailureManager;
import org.apache.ignite.internal.failure.FailureType;
import org.apache.ignite.internal.fileio.FileIoFactory;
import org.apache.ignite.internal.lang.IgniteInternalCheckedException;
import org.apache.ignite.internal.lang.IgniteInternalException;
import org.apache.ignite.internal.lang.IgniteStringFormatter;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.pagememory.persistence.GroupPartitionId;
import org.apache.ignite.internal.pagememory.persistence.PageReadWriteManager;
import org.apache.ignite.internal.pagememory.persistence.store.FilePageStore;
import org.apache.ignite.internal.pagememory.persistence.store.FilePageStoreFactory;
import org.apache.ignite.internal.pagememory.persistence.store.GroupPageStoresMap;
import org.apache.ignite.internal.pagememory.persistence.store.LongOperationAsyncExecutor;
import org.apache.ignite.internal.pagememory.persistence.store.PageStore;
import org.apache.ignite.internal.pagememory.util.PageIdUtils;
import org.apache.ignite.internal.util.IgniteUtils;
import org.jetbrains.annotations.Nullable;

public class FilePageStoreManager
implements PageReadWriteManager {
    private static final IgniteLogger LOG = Loggers.forClass(FilePageStoreManager.class);
    public static final String FILE_SUFFIX = ".bin";
    public static final String TMP_FILE_SUFFIX = ".tmp";
    public static final String DEL_FILE_SUFFIX = ".del";
    public static final String PART_FILE_PREFIX = "part-";
    public static final String PART_DELTA_FILE_PREFIX = "part-%d-delta-";
    public static final String PART_FILE_TEMPLATE = "part-%d.bin";
    public static final String DEL_PART_FILE_TEMPLATE = "part-%d.del";
    public static final String DEL_PART_FILE_REGEXP = "part-(\\d+).del";
    public static final String PART_DELTA_FILE_TEMPLATE = "part-%d-delta-%d.bin";
    public static final String TMP_PART_DELTA_FILE_TEMPLATE = "part-%d-delta-%d.bin.tmp";
    public static final String GROUP_DIR_PREFIX = "table-";
    private final Path dbDir;
    private final LongOperationAsyncExecutor cleanupAsyncExecutor;
    private final GroupPageStoresMap<FilePageStore> groupPageStores;
    private final FilePageStoreFactory filePageStoreFactory;
    private final FailureManager failureManager;

    public FilePageStoreManager(String igniteInstanceName, Path storagePath, FileIoFactory filePageStoreFileIoFactory, int pageSize, FailureManager failureManager) {
        this.dbDir = storagePath.resolve("db");
        this.failureManager = failureManager;
        this.cleanupAsyncExecutor = new LongOperationAsyncExecutor(igniteInstanceName, LOG);
        this.groupPageStores = new GroupPageStoresMap(this.cleanupAsyncExecutor);
        this.filePageStoreFactory = new FilePageStoreFactory(filePageStoreFileIoFactory, pageSize);
    }

    public void start() throws IgniteInternalCheckedException {
        String tmpDir;
        try {
            Files.createDirectories(this.dbDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IgniteInternalCheckedException("Could not create work directory for page stores: " + String.valueOf(this.dbDir), (Throwable)e);
        }
        if (LOG.isWarnEnabled() && (tmpDir = System.getProperty("java.io.tmpdir")) != null && this.dbDir.startsWith(tmpDir)) {
            LOG.warn("Persistence store directory is in the temp directory and may be cleaned. To avoid this change location of persistence directories [currentDir={}]", new Object[]{this.dbDir});
        }
        ArrayList toDelete = new ArrayList();
        try (Stream<Path> tmpFileStream = Files.find(this.dbDir, Integer.MAX_VALUE, (path, basicFileAttributes) -> path.getFileName().toString().endsWith(TMP_FILE_SUFFIX), new FileVisitOption[0]);){
            toDelete.addAll(tmpFileStream.collect(Collectors.toList()));
        }
        catch (IOException e) {
            throw new IgniteInternalCheckedException("Error while searching temporary files:" + String.valueOf(this.dbDir), (Throwable)e);
        }
        Pattern delPartitionFilePattern = Pattern.compile(DEL_PART_FILE_REGEXP);
        try (Stream<Path> delFileStream = Files.find(this.dbDir, Integer.MAX_VALUE, (path, basicFileAttributes) -> path.getFileName().toString().endsWith(DEL_FILE_SUFFIX), new FileVisitOption[0]);){
            delFileStream.forEach(delFilePath -> {
                Matcher matcher = delPartitionFilePattern.matcher(delFilePath.getFileName().toString());
                if (!matcher.matches()) {
                    throw new IgniteInternalException("Unknown file: " + String.valueOf(delFilePath));
                }
                Path tableWorkDir = delFilePath.getParent();
                int partitionId = Integer.parseInt(matcher.group(1));
                toDelete.add(tableWorkDir.resolve(String.format(PART_FILE_TEMPLATE, partitionId)));
                try {
                    toDelete.addAll(List.of(FilePageStoreManager.findPartitionDeltaFiles(tableWorkDir, partitionId)));
                }
                catch (IgniteInternalCheckedException e) {
                    throw new IgniteInternalException("Error when searching delta files for partition:" + String.valueOf(delFilePath), (Throwable)e);
                }
                toDelete.add(delFilePath);
            });
        }
        catch (IOException e) {
            throw new IgniteInternalCheckedException("Error while searching temporary files:" + String.valueOf(this.dbDir), (Throwable)e);
        }
        if (!toDelete.isEmpty()) {
            LOG.info("Files to be deleted: {}", new Object[]{toDelete});
            toDelete.forEach(IgniteUtils::deleteIfExists);
        }
    }

    public void stop() throws Exception {
        this.stopAllGroupFilePageStores(false);
        this.cleanupAsyncExecutor.awaitAsyncTaskCompletion(false);
    }

    @Override
    public void read(int grpId, long pageId, ByteBuffer pageBuf, boolean keepCrc) throws IgniteInternalCheckedException {
        try {
            FilePageStore pageStore = this.getStoreWithCheckExists(new GroupPartitionId(grpId, PageIdUtils.partitionId(pageId)));
            pageStore.read(pageId, pageBuf, keepCrc);
        }
        catch (IgniteInternalCheckedException e) {
            this.failureManager.process(new FailureContext(FailureType.CRITICAL_ERROR, (Throwable)e));
            throw e;
        }
    }

    @Override
    public PageStore write(int grpId, long pageId, ByteBuffer pageBuf, boolean calculateCrc) throws IgniteInternalCheckedException {
        try {
            FilePageStore pageStore = this.getStoreWithCheckExists(new GroupPartitionId(grpId, PageIdUtils.partitionId(pageId)));
            pageStore.write(pageId, pageBuf, calculateCrc);
            return pageStore;
        }
        catch (IgniteInternalCheckedException e) {
            this.failureManager.process(new FailureContext(FailureType.CRITICAL_ERROR, (Throwable)e));
            throw e;
        }
    }

    @Override
    public long allocatePage(int grpId, int partId, byte flags) throws IgniteInternalCheckedException {
        assert (partId >= 0 && partId <= 65500) : partId;
        try {
            FilePageStore pageStore = this.getStoreWithCheckExists(new GroupPartitionId(grpId, partId));
            int pageIdx = pageStore.allocatePage();
            return PageIdUtils.pageId(partId, flags, pageIdx);
        }
        catch (IgniteInternalCheckedException e) {
            this.failureManager.process(new FailureContext(FailureType.CRITICAL_ERROR, (Throwable)e));
            throw e;
        }
    }

    public Stream<GroupPageStoresMap.GroupPartitionPageStore<FilePageStore>> allPageStores() {
        return this.groupPageStores.getAll();
    }

    @Nullable
    public FilePageStore getStore(GroupPartitionId groupPartitionId) {
        return this.groupPageStores.get(groupPartitionId);
    }

    private FilePageStore getStoreWithCheckExists(GroupPartitionId groupPartitionId) throws IgniteInternalCheckedException {
        FilePageStore filePageStore = this.getStore(groupPartitionId);
        if (filePageStore == null) {
            throw new IgniteInternalCheckedException(IgniteStringFormatter.format((String)"Partition file page store is either not initialized or deleted: [groupId={}, partitionId={}]", (Object[])new Object[]{groupPartitionId.getGroupId(), groupPartitionId.getPartitionId()}));
        }
        return filePageStore;
    }

    void stopAllGroupFilePageStores(boolean cleanFiles) {
        List partitionPageStores = this.groupPageStores.getAll().map(GroupPageStoresMap.GroupPartitionPageStore::pageStore).collect(Collectors.toList());
        this.groupPageStores.clear();
        Runnable stopPageStores = () -> {
            try {
                FilePageStoreManager.stopGroupFilePageStores(partitionPageStores, cleanFiles);
                LOG.info("Cleanup cache stores [total={}, cleanFiles={}]", new Object[]{partitionPageStores.size(), cleanFiles});
            }
            catch (Exception e) {
                LOG.info("Failed to gracefully stop page store managers", (Throwable)e);
            }
        };
        if (cleanFiles) {
            this.cleanupAsyncExecutor.async(stopPageStores, "file-page-stores-cleanup");
        } else {
            stopPageStores.run();
        }
    }

    private static void stopGroupFilePageStores(List<FilePageStore> partitionPageStores, boolean cleanFiles) throws IgniteInternalCheckedException {
        try {
            List closePageStores = partitionPageStores.stream().map(pageStore -> () -> pageStore.stop(cleanFiles)).collect(Collectors.toList());
            IgniteUtils.closeAll(closePageStores);
        }
        catch (IgniteInternalCheckedException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IgniteInternalCheckedException((Throwable)e);
        }
    }

    private Path ensureGroupWorkDir(int groupId) throws IgniteInternalCheckedException {
        Path groupWorkDir = this.groupDir(groupId);
        try {
            Files.createDirectories(groupWorkDir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IgniteInternalCheckedException("Failed to initialize group working directory (failed to create, make sure the work folder has correct permissions): " + String.valueOf(groupWorkDir), (Throwable)e);
        }
        return groupWorkDir;
    }

    static Path[] findPartitionDeltaFiles(Path groupWorkDir, int partitionId) throws IgniteInternalCheckedException {
        String partitionDeltaFilePrefix = String.format(PART_DELTA_FILE_PREFIX, partitionId);
        File[] files = groupWorkDir.toFile().listFiles((dir, name) -> name.startsWith(partitionDeltaFilePrefix));
        assert (files != null) : groupWorkDir;
        return (Path[])Stream.of(files).map(File::toPath).toArray(Path[]::new);
    }

    public Path tmpDeltaFilePageStorePath(int groupId, int partitionId, int index) {
        return this.groupDir(groupId).resolve(String.format(TMP_PART_DELTA_FILE_TEMPLATE, partitionId, index));
    }

    public Path deltaFilePageStorePath(int groupId, int partitionId, int index) {
        return this.groupDir(groupId).resolve(String.format(PART_DELTA_FILE_TEMPLATE, partitionId, index));
    }

    public CompletableFuture<Void> destroyPartition(GroupPartitionId groupPartitionId) {
        FilePageStore removed = this.groupPageStores.remove(groupPartitionId);
        assert (removed != null) : IgniteStringFormatter.format((String)"Parallel deletion is not allowed: [groupId={}, partitionId={}]", (Object[])new Object[]{groupPartitionId.getGroupId(), groupPartitionId.getPartitionId()});
        assert (removed.isMarkedToDestroy()) : IgniteStringFormatter.format((String)"Wasn't marked for deletion: [groupId={}, partitionId={}]", (Object[])new Object[]{groupPartitionId.getGroupId(), groupPartitionId.getPartitionId()});
        return this.cleanupAsyncExecutor.async(() -> {
            Path partitionDeleteFilePath = Files.createFile(this.groupDir(groupPartitionId.getGroupId()).resolve(String.format(DEL_PART_FILE_TEMPLATE, groupPartitionId.getPartitionId())), new FileAttribute[0]);
            removed.stop(true);
            Files.delete(partitionDeleteFilePath);
        }, "destroy-group-" + groupPartitionId.getGroupId() + "-partition-" + groupPartitionId.getPartitionId());
    }

    public FilePageStore readOrCreateStore(GroupPartitionId groupPartitionId, ByteBuffer readBuffer) throws IgniteInternalCheckedException {
        Path tableWorkDir = this.ensureGroupWorkDir(groupPartitionId.getGroupId());
        Path partFilePath = tableWorkDir.resolve(String.format(PART_FILE_TEMPLATE, groupPartitionId.getPartitionId()));
        Path[] partDeltaFiles = FilePageStoreManager.findPartitionDeltaFiles(tableWorkDir, groupPartitionId.getPartitionId());
        return this.filePageStoreFactory.createPageStore(readBuffer.rewind(), partFilePath, partDeltaFiles);
    }

    public void addStore(GroupPartitionId groupPartitionId, FilePageStore filePageStore) {
        this.groupPageStores.compute(groupPartitionId, oldFilePageStore -> {
            assert (oldFilePageStore == null) : groupPartitionId;
            return filePageStore;
        });
    }

    public void destroyGroupIfExists(int groupId) throws IOException {
        Path groupDir = this.groupDir(groupId);
        try {
            if (Files.exists(groupDir, new LinkOption[0])) {
                IgniteUtils.deleteIfExistsThrowable((Path)groupDir);
            }
        }
        catch (IOException e) {
            throw new IOException("Failed to delete group directory: " + String.valueOf(groupDir), e);
        }
    }

    private Path groupDir(int groupId) {
        return this.dbDir.resolve(GROUP_DIR_PREFIX + groupId);
    }
}

