/*
 * Copyright 2012 Splunk, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"): you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.splunk;

/**
 * The {@code CollectionArgs} class contains arguments for retrieving and 
 * listing entities from a collection, such as the number of entities to return 
 * and how to sort them.
 */
public class CollectionArgs extends Args {
    
    /**
     * Indicates whether to sort entries in ascending or descending order.
     */
    public static enum SortDirection {
        /** Sort entries in ascending order. */
        ASC("asc"),
        /** Sort entries in descending order. */
        DESC("desc");
        
        private String value;
        
        private SortDirection(String value) {
            this.value = value;
        }
        
        /**
         * @return The REST API value for this enumerated constant.
         */
        public String toString() {
            return this.value;
        }
    }
    
    /**
     * Indicates the sorting mode for entries.
     */
    public static enum SortMode {
        /**
         * If all values of the field are numbers, collate numerically.
         * Otherwise, collate alphabetically. 
         */
        AUTO("auto"),
        /** Collate alphabetically. */
        ALPHA("alpha"),
        /** Collate alphabetically, case-sensitive. */
        ALPHA_CASE("alpha_case"),
        /** Collate numerically. */
        NUM("num");
        
        private String value;
        
        private SortMode(String value) {
            this.value = value;
        }
        
        /**
         * @return The REST API value for this enumerated constant.
         */
        public String toString() {
            return this.value;
        }
    }
    
    /**
     * Class constructor.
     */
    public CollectionArgs() { super(); }
    
    /* BEGIN AUTOGENERATED CODE */
    
    /**
     * Sets the app context in which to list the collection.
     * 
     * @param app
     *      The app context in which to list the collection. A {@code null} value indicates no app context, and a value of {@code "-"} indicates an app wildcard.
     */
    public void setApp(String app) {
        this.put("app", app);
    }
    
    /**
     * Sets the owner context in which to list the collection.
     * 
     * @param owner
     *      The owner context in which to list the collection. A value of {@code "-"} indicates a wildcard, and a {@code null} value indicates no owner context.
     */
    public void setOwner(String owner) {
        this.put("owner", owner);
    }
    
    /**
     * Sets the sharing context in which to list the collection.
     * 
     * @param sharing
     *      The sharing context in which to list the collection. Valid values are "user", "app", "global", and "system".
     */
    public void setSharing(String sharing) {
        this.put("sharing", sharing);
    }
    
    /**
     * Sets the maximum number of entries to return.
     * 
     * @param count
     *      The maximum number of entries to return. To return all entries, specify 0.
     */
    public void setCount(int count) {
        this.put("count", count);
    }
    
    /**
     * Sets the index of the first entry to return.
     * 
     * @param offset
     *      The index of the first entry to return.
     */
    public void setOffset(int offset) {
        this.put("offset", offset);
    }
    
    /**
     * Sets a search query to filter the response. The response matches field values against the search query. For example, "foo" matches any object that has "foo" as a substring in a field, and "field_name=field_value" restricts the match to a single field.
     * 
     * @param search
     *      A search query to filter the response. 
     */
    public void setSearch(String search) {
        this.put("search", search);
    }
    
    /**
     * Sets the direction to sort entries.
     * 
     * @param sortDirection
     *      The sorting order--ascending or descending.
     */
    public void setSortDirection(SortDirection sortDirection) {
        this.put("sort_dir", sortDirection);
    }
    
    /**
     * Sets the field to use for sorting. 
     * 
     * @param sortKey
     *      The field to sort by.
     */
    public void setSortKey(String sortKey) {
        this.put("sort_key", sortKey);
    }
    
    /**
     * Sets the mode to use for sorting.
     * 
     * @param sortMode
     *      The collating sequence for sorting entries.
     */
    public void setSortMode(SortMode sortMode) {
        this.put("sort_mode", sortMode);
    }
    
    /* END AUTOGENERATED CODE */
}
